package mcfall.raytracer;

import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.LinkedList;

import javax.swing.Box;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.text.BadLocationException;
import javax.swing.text.Caret;
import javax.swing.text.Document;
import javax.swing.text.Highlighter;
import javax.swing.text.JTextComponent;
import javax.swing.text.Position;
import javax.swing.text.Segment;

public class SearchPanel extends JPanel {
	private JLabel findLabel;
	private JTextField searchField;
	private JButton findButton;
	private JButton findNextButton;
	
	private JLabel directionLabel;
	private ButtonGroup directionGroup;
	private JRadioButton searchBackButton;
	private JRadioButton searchForwardButton;
	private Box contentBox;
	
	private LinkedList<javax.swing.text.Position> searchPositions;
	private int currentStartIndex;
	private Segment currentSegment;
	char[] buffer = new char[1024];
	private Document document;
	private JTextComponent component;
	private String currentSearchText;
	
	enum SearchDirection {FORWARD, BACKWARD};
	
	private SearchDirection currentDirection;
	
	public SearchPanel (JTextComponent component) {
		this.component = component;
		this.document = component.getDocument();
		
		findLabel = new JLabel ("Search for: ");
		searchField = new JTextField (25);
		findButton = new JButton ("Find");
		findNextButton = new JButton ("Find next");
		findNextButton.setEnabled(false);
		
		directionLabel = new JLabel ("Direction:");
		
		searchBackButton = new JRadioButton ("Backward");
		searchForwardButton = new JRadioButton ("Forward");
		
		directionGroup = new ButtonGroup ();
		directionGroup.add(searchForwardButton);
		directionGroup.add(searchBackButton);
		currentDirection = SearchDirection.FORWARD;
		searchForwardButton.setSelected(true);
		
		contentBox = Box.createHorizontalBox();
		contentBox.add (findLabel);
		contentBox.add(searchField);
		contentBox.add(findButton);
		contentBox.add(findNextButton);
		
		contentBox.add(searchForwardButton);
		contentBox.add(searchBackButton);
		
		add (contentBox);
		
		searchPositions = new LinkedList<Position> ();
		
		//currentSegment = new Segment (buffer, 0, buffer.length);
		currentSegment = new Segment ();
		currentSegment.setPartialReturn(true);
		setupHandlers ();
	}
	
	private void findNextAndUpdateGUI () {
		boolean found = findForward();
		findNextButton.setEnabled(found);
		if (!found) {
			Toolkit.getDefaultToolkit().beep();
		}	
	}
	
	private void setupHandlers () {
		findButton.addActionListener(new ActionListener () {
			public void actionPerformed(ActionEvent e) {
				searchPositions.clear();
				currentStartIndex = 0;
				currentSegment = new Segment ();
				currentSegment.setPartialReturn(true);
				currentSearchText = searchField.getText();
				findNextAndUpdateGUI ();
			}			
		});
		
		findNextButton.addActionListener(new ActionListener () {
			public void actionPerformed (ActionEvent e) {
				findNextAndUpdateGUI();
			}
		});
		
		searchForwardButton.addActionListener(new ActionListener () {
			public void actionPerformed(ActionEvent e) {
				currentDirection = SearchDirection.FORWARD;			
			}			
		});
		
		searchBackButton.addActionListener(new ActionListener () {
			public void actionPerformed(ActionEvent e) {
				currentDirection = SearchDirection.BACKWARD;
			}
		});
	}
	
	/**
	 * Locates and highlights the next occurrence of the current search text
	 * @return true if an instance of the search text was found after the current instance,
	 * false otherwise
	 */
	private boolean findForward () {
		//  Starting at the current index, read in a chunk of the document and look for the search text until we've found it
		int nleft = document.getLength()-currentStartIndex;
						     
		while (nleft > 0) {
			try {
				document.getText(currentStartIndex, nleft, currentSegment);
			} catch (BadLocationException e) { e.printStackTrace(); }
			
			int index = findIndexInCurrentSegment (currentSearchText);
			
			if (index > 0) {
				updateSelection(index + currentStartIndex);
				return true;				
			}
			nleft -= currentSegment.count;
			currentStartIndex += currentSegment.count;
		}
		
		//  At this point, we didn't find it
		return false;		
	}

	private void updateSelection(final int actualIndex) {
		SwingUtilities.invokeLater( new Runnable () {
			public void run () {
				Caret caret = component.getCaret();
				caret.setDot(actualIndex);
				caret.moveDot(actualIndex+searchField.getText().length());
				caret.setSelectionVisible(true);						
				component.repaint();
			}
		});
	}
		
	/**
	 * This method finds the index of the first location of a given string in 
	 * the current segment.  For now, take the easy way out and construct a string.  Later on, 
	 * should maybe do an iteration through the characters to avoid the cost of string construction
	 * @return
	 */
	private int findIndexInCurrentSegment (String text) {
		int currentSearchIndex = 0;
		
		char currentContentCharacter = currentSegment.current();
		
		while (currentSearchIndex < text.length() && currentContentCharacter != Segment.DONE) {
			if (currentContentCharacter == text.charAt(currentSearchIndex)) {
				currentSearchIndex++;
				if (currentSearchIndex == text.length()) {
					return currentSegment.getIndex()-text.length()+1;
				}
			}
			else {
				if (currentContentCharacter == text.charAt(0)) {
					currentSearchIndex = 1;
				}
				else {
					currentSearchIndex = 0;
				}
				
			}
			currentContentCharacter = currentSegment.next();
		}
		
		return -1;
	}
}
